// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "degub.h"
#include "container.h"
#include "recompiler.h"
#include "opcode.h"
#include "messages.h"
#include "stringCommons.h"
#include <fstream>

#define MAX_INTERNAL_DYNAREC_ERROR 11 //16:55 2006-09-17

#define TRANSLATE_IA(eia) (g::cache_enabled ? eia : PHYSICALIZE(recmem.translate_i(eia)))
#define RDEGUB if(g::rec_log) DEGUB
#define VRDEGUB if(g::verbose) RDEGUB

//mov eax, eia2pos(cia)
void Recompiler::_unimplemented_opcode() {
	AB(0xB8);
	AD(eia2pos(cia));
	add_fatal_error(REC_UNIMPLEMENTED_OPCODE);
	/*last_undefined = true;*/
}

Recompiler::Recompiler(CORE_ARGUMENTS a) :
CoreBase(recmem, a), rcarr(0, 5), memory(recmem.pmem()), hWnd(a.hWnd),
recmem(h, r, interrupt), cache_memory(recmem.pcmem()), //cur_tdrb(NULL),
total_rec_cycles(0), ipage_faults(0), m_staticblock(createStaticBlock()) {
	DWORD count=0;

#define ADD_REC_OPCODE(asm, pri, sm, sec) count++;\
	rcarr.addOpcode(&Recompiler::_##asm, #asm, pri, sm, sec);
	OPCODES(ADD_REC_OPCODE);
	rcarr.addOpcode(&Recompiler::_addic_, "addic.", 13, 0, 0); count++;
	rcarr.addOpcode(&Recompiler::_stwcx_, "stwcx.", 31, 21, 150); count++;
	DEGUB("Supported opcodes: %i\n", count);

#define ADD_REC_UNIMPLEMENTED_OPCODE(asm, pri, sm, sec) count++;\
	rcarr.addOpcode(&Recompiler::_unimplemented_opcode, #asm, pri, sm, sec);
	UNIMPLEMENTED_OPCODES(ADD_REC_UNIMPLEMENTED_OPCODE);
	DEGUB("Defined opcodes: %i\n", count);

	//rcarr.dump();	//temporary

	//Intel FPU Init; incomplete?
	//control87(_PC_64 | _RC_NEAR | _DN_SAVE, _MCW_PC | _MCW_RC | _MCW_DN);
}


Recompiler::~Recompiler() {
	if(g::advanced_mode) {
		DEGUB("%i Instruction Page Faults\n", ipage_faults);
	}
}

bool Recompiler::handle_load() {
	memcpy(cache_memory, memory, CACHE_SIZE);
	if(g::degub_run) {
		recmem.degub();
		r.degub(false, old_registers);
	}

	if(g::verbose) {
		DEGUB("Main memory @ 0x%08X  Cache memory @ 0x%08X\n", memory, cache_memory);
	}

	Timing timing("Initial recompilation");
	TGLE(recompile(entry_point, BAD_ADDRESS));
	//TGLE(complete_recompilation());
	nia_pos = drbv_eia2pos(entry_point);
	MYASSERT(nia_pos != 0);

	return true;
}

void Recompiler::_request_label(std::vector<LABEL> &_lv,
																std::vector<LABEL> &_lrv, const char *label)
{
	MYASSERT(cur_recbuf != NULL);
	for(size_t i=0; i<_lv.size(); i++) {
		if(strcmp(_lv[i].label, label) == 0) {
			if((int(_lv[i].pos - POS - 1) > 127) && (int(_lv[i].pos - POS - 1) < -128)) {
				DEGUB("Label length: %i\n", int(_lv[i].pos - POS - 1));
				throw rec_fatal_exception("Too long label request");
			}
			AB(_lv[i].pos - POS - 1);
			return;
		}
	}
	//it hasn't been added yet...
	LABEL l = { POS, label };
	_lrv.push_back(l);
	AB(0);
}
void Recompiler::_add_label(std::vector<LABEL> &_lv,
														std::vector<LABEL> &_lrv, const char *label)
{
	LABEL l = { POS, label };
	for(size_t i=0; i<_lv.size(); i++) {
		if(strcmp(_lv[i].label, label) == 0) {
			DEGUB("Dublicate label: \"%s\"\n", label);
			throw rec_fatal_exception("Duplicate label");
		}
	}
	_lv.push_back(l);
	for(size_t i=0; i<_lrv.size(); i++) {
		if(strcmp(_lrv[i].label, label) == 0) {
			if((int(POS - _lrv[i].pos - 1) < 127) && (int(POS - _lrv[i].pos - 1) < -128)) {
				DEGUB("Label length: %i\n", int(POS - _lrv[i].pos - 1));
				throw rec_fatal_exception("Too long label add");
			}
			cur_recbuf->rcb()[_lrv[i].pos] = BYTE(POS - _lrv[i].pos - 1);
			_lrv.erase(_lrv.begin() + i);
			i--;
		}
	}
}

void Recompiler::request_label(const char *label) {
	_request_label(lv, lrv, label);
}
void Recompiler::add_label(const char *label) {
	_add_label(lv, lrv, label);
}
void Recompiler::request_internal_label(const char *label) {
	_request_label(ilv, ilrv, label);
}
void Recompiler::add_internal_label(const char *label) {
	_add_label(ilv, ilrv, label);
}
void Recompiler::clear_internal_labels() {
	if(!ilrv.empty()) {
		for(size_t i=0; i<ilrv.size(); i++) {
			DEGUB("Internal label request: \"%s\" @ pos %i\n",
				ilrv[i].label, ilrv[i].pos);
		}
		throw rec_fatal_exception("Internal label request!");
	}
	ilv.clear();
	ilrv.clear();
}

DWORD Recompiler::eia2pos(DWORD eia) {
	MYASSERT(cur_recbuf != NULL);
	//if eia isn't in the already recompiled section of the current block
	if(!(eia >= cur_recbuf->start_eia &&
		eia < cur_recbuf->start_eia + cur_recbuf->rcbtb.size()*4))
	{
		//size_t index = getDRBIndexRaw(eia);
		//if(index == -1)
		size_t index = findDRBIndexByEIA(eia);
		if(index != -1)
			return drbv[index].rcbtb()[(eia - drbv[index].start_eia()) / 4];
	}
	IAPOS iapos = { eia, POS };
	cur_recbuf->trv.push_back(iapos);
	return 0; //I think this is a good invalid value.
}
DWORD Recompiler::raw_eia2pos(DWORD eia) {
	//MYASSERT(cur_recbuf != NULL);
	size_t index = getDRBIndexRaw(eia);
	if(index == -1)
		return 0;
	else
		return drbv[index].eia2pos(eia);
}
DWORD Recompiler::drbv_eia2pos(DWORD eia) {
	MYASSERT(!cur_recbuf);
	size_t index = findDRBIndexByEIA(eia);
	if(index == -1)
		return 0;
	else
		return drbv[index].eia2pos(eia);
}

bool Recompiler::addcia2pos() {
	MYASSERT(cur_recbuf != NULL);
	MYASSERT(cia >= cur_recbuf->start_eia);
	cur_recbuf->rcbtb.push_back(POS);
	return true;
}

size_t Recompiler::findDRBIndexByEIA(DWORD eia) const {
	//Find the block this address is in.
	//Linear search. If there are many blocks (like >50), a binary search may be faster.
	//We'd need to make sure the drbv is sorted then.
	for(size_t i=0; i<drbv.size(); i++) {
		if(EIA_IS_IN_INDEX(eia, i))
			return i;
	}
	return BAD_ADDRESS;
}
size_t Recompiler::findDRBIndexByPIA(DWORD pia) const {
	for(size_t i=0; i<drbv.size(); i++) {
		if(PIA_IS_IN_INDEX(pia, i))
			return i;
	}
	return BAD_ADDRESS;
}
size_t Recompiler::findDRBIndexByPos(DWORD pos) const {
	for(size_t i=0; i<drbv.size(); i++) {
		if(pos >= DWORD(drbv[i].rcb()) &&
			pos < DWORD(drbv[i].rcb()) + drbv[i].recbuf_size())
			return i;
	}
	return BAD_ADDRESS;
}
size_t Recompiler::getDRBIndexRaw(DWORD eia) {
	DWORD temp_opcode = recmem.read_instruction(eia);
	recmem.clear_degub();
	return getDRBIndexRaw_base(eia, temp_opcode);
}
size_t Recompiler::getDRBIndexRaw_base(DWORD eia, DWORD temp_opcode) {
	if(IS_RECOMPILED_OPCODE_TYPE1(temp_opcode)) {
		size_t index = GET_DRBINDEX(temp_opcode);
		if(index >= drbv.size())
			throw rec_fatal_exception("Internal DynaRec error no.5");
		if(!EIA_IS_IN_INDEX(eia, index))
			throw rec_fatal_exception("Internal DynaRec error no.6");
		return index;
	} else if(IS_RECOMPILED_OPCODE_TYPE2(temp_opcode)) {
		size_t index = GET_DRBINDEX(temp_opcode);
		if(index >= m_type2.size())
			throw rec_fatal_exception("Internal DynaRec error no.7");
		vector<size_t> &v = m_type2[index];
		for(size_t i=0; i<v.size(); i++) {
			if(EIA_IS_IN_INDEX(eia, v[i]))
				return v[i];
		}
		throw rec_fatal_exception("Internal DynaRec error no.8");
	}
	return BAD_ADDRESS;
}
size_t Recompiler::getDRBIndexRawPIA_base(DWORD pia, DWORD temp_opcode) {
	MYASSERT(IS_RECOMPILED_OPCODE_ANYTYPE(temp_opcode));
	if(IS_RECOMPILED_OPCODE_TYPE1(temp_opcode)) {
		size_t index = GET_DRBINDEX(temp_opcode);
		if(index >= drbv.size())
			throw rec_fatal_exception("Internal DynaRec error no.5");
		if(!PIA_IS_IN_INDEX(pia, index))
			throw rec_fatal_exception("Internal DynaRec error no.6");
		return index;
	} else if(IS_RECOMPILED_OPCODE_TYPE2(temp_opcode)) {
		size_t index = GET_DRBINDEX(temp_opcode);
		if(index >= m_type2.size())
			throw rec_fatal_exception("Internal DynaRec error no.7");
		vector<size_t> &v = m_type2[index];
		for(size_t i=0; i<v.size(); i++) {
			if(PIA_IS_IN_INDEX(pia, v[i]))
				return v[i];
		}
		throw rec_fatal_exception("Internal DynaRec error no.8");
	}
	return BAD_ADDRESS;
}
DWORD Recompiler::getPPCCode(DWORD pia) {
	DWORD temp_opcode = recmem.read_pia(pia);
	if(IS_RECOMPILED_OPCODE_ANYTYPE(temp_opcode)) {
		const DynaRecBlock &drb = drbv[getDRBIndexRawPIA_base(pia, temp_opcode)];
		return drb.ppc_code()[(pia - drb.start_pia()) / 4];
	} else
		return temp_opcode;
}

typedef int (__cdecl *comparefunc)(const void *, const void *);

int __cdecl DWORDcompare(const DWORD *arg1, const DWORD *arg2) {
	return (*arg1 < *arg2) ? -1 : ((*arg1 > *arg2) ? 1 : 0);
}

DWORD Recompiler::reverse_rcbt(DWORD rcbpos) const {
	size_t i = findDRBIndexByPos(rcbpos);
	if(i == BAD_ADDRESS) {
		DEGUB("%08X not foind in reverse_rcbt()\n", rcbpos);
		//throw logic_error("rrcbt1");
		THE_ONE_MESSAGE_BOX("Internal DynaRec error no.1");
		return BAD_ADDRESS;
	}
	const DynaRecBlock &drb = drbv[i];
	void *result = bsearch(&rcbpos, drb.rcbtb(),
		drb.rcbtb_size(), sizeof(DWORD), (comparefunc)DWORDcompare);
	if(!result) {
		DEGUB("%08X not foind in reverse_rcbt()\n", rcbpos);
		DEGUB("%i DWORDs in rcbtb %i\n", drb.rcbtb_size(), i);
		DWORD old = 0;
		for(size_t j=0; j<drb.rcbtb_size(); j++) {
			DEGUB("0x%08X %08X\n", drb.start_eia() + j*4, drb.rcbtb()[j]);
			if(drb.rcbtb()[j] <= old) {
				DEGUB("ERROR!!!\n");
				break;
			}
			old = drb.rcbtb()[j];
		}
		throw rec_fatal_exception("Internal DynaRec error no.2");
	} else
		return DWORD(result) - DWORD(drb.rcbtb()) + drb.start_eia();
}

string Recompiler::getdasma(DWORD address) {
	return getdasmo(address, recmem.read_pia(address));
}
string Recompiler::getdasmo(DWORD cia, DWORD opcode) {
	disassembly::opcode = opcode;
	disassembly::cia = cia;
	const char *opasm = rcarr.getOpAsm(disassembly::opcode);
	ostringstream str;
	if(opasm) {
		Container<char> buffer(90);
		(*asmarr.getOpFunc(disassembly::opcode))(buffer);
		str << opasm << buffer;
	}
	return str.str();
}

bool Recompiler::next_instruction_is_static_branch() {
	MYASSERT(cur_recbuf != NULL);
	DWORD opcode = recmem.read_instruction(nia);	//this one may throw a page fault exception
	recmem.clear_degub();
	void (Recompiler::*opfunc)() = rcarr.getOpFunc(opcode);
	return (opfunc == &Recompiler::_b) || (opfunc == &Recompiler::_bc);
}

void Recompiler::add_drb2overwrite(DWORD pia, size_t new_index) {
	DWORD temp_opcode = recmem.read_pia(pia);
	if(IS_RECOMPILED_OPCODE_TYPE1(temp_opcode)) {
		vector<size_t> v;
		v.push_back(GET_DRBINDEX(temp_opcode));
		v.push_back(new_index);

		DWORD new_opcode = MAKE_RECOMPILED_OPCODE_TYPE2(m_type2.size());
		recmem.write_pia(pia, new_opcode);
		m_type2.push_back(v);
		VRDEGUB("Changed type1(%08X) to type2(%08X, %i %i) @ %08X | cycle %I64i\n",
			temp_opcode, new_opcode, v[0], v[1], pia, cc.cycles.QuadPart);
	} else if(IS_RECOMPILED_OPCODE_TYPE2(temp_opcode)) {
		size_t index = GET_DRBINDEX(temp_opcode);
		if(index >= m_type2.size()) {
			throw rec_fatal_exception("Internal DynaRec error no.4");
		}
		vector<size_t> &v = m_type2[index];
		v.push_back(new_index);
		VRDEGUB("Added DRB %i to type2(%08X) @ %08X. New size: %i | cycle %I64i\n",
			new_index, temp_opcode, pia, v.size(), cc.cycles.QuadPart);
	} else {
		DWORD new_opcode = MAKE_RECOMPILED_OPCODE_TYPE1(new_index);
		recmem.write_pia(pia, new_opcode);
		if(g::rec_disasm) {
			VRDEGUB("Overwrote opcode %08X @ 0x%08X with %08X | cycle %I64i\n",
				temp_opcode, pia, new_opcode, cc.cycles.QuadPart);
		}
	}
}
void Recompiler::remove_drb_from_overwrite(DWORD pia, size_t drb_index) {
	//assert that drb_index exists in pia
	//no... if another block occupying the same pia is cleared, the assertion will fail

	DWORD temp_opcode = recmem.read_pia(pia);
	if(IS_RECOMPILED_OPCODE_TYPE1(temp_opcode)) {
		//MYASSERT(GET_DRBINDEX(temp_opcode) == drb_index);
		if(GET_DRBINDEX(temp_opcode) == drb_index) {
			DynaRecBlock drb = drbv[drb_index];
			DWORD old_opcode = drb.ppc_code()[(pia - drb.start_pia()) / 4];
			recmem.write_pia(pia, old_opcode);
			DEGUB("Restored opcode %08X @ 0x%08X with %08X | cycle %I64i\n",
				temp_opcode, pia, old_opcode, cc.cycles.QuadPart);
		}
	} else if(IS_RECOMPILED_OPCODE_TYPE2(temp_opcode)) {
		size_t index2 = GET_DRBINDEX(temp_opcode);
		if(index2 >= m_type2.size()) {
			throw rec_fatal_exception("Internal DynaRec error no.10");
		}
		vector<size_t> &v = m_type2[index2];
		bool found = false;
		for(size_t i=0; i<v.size(); i++) {
			if(v[i] == drb_index) {
				found = true;
				v.erase(v.begin() + i);
				break;
			}
		}
		if(found) {
			/*if(v.size == 1) {
			DWORD new_opcode = MAKE_RECOMPILED_OPCODE_TYPE2(v[0]);
			recmem.write_pia(pia, new_opcode);
			VDEGUB("Changed type2(%08X, %i %i) to type1(%08X) @ %08X | cycle %I64i\n",
			temp_opcode, v[0], drb_index, new_opcode, pia, cc.cycles);
			}*/	//we can't do this 'cause it'd require remaking of all type2 overwrites
			VRDEGUB("Removed DRB %i from type2(%08X) @ %08X. New size: %i | cycle %I64i\n",
				drb_index, temp_opcode, pia, v.size(), cc.cycles.QuadPart);
			MYASSERT(v.size() != 0);
		}
	} else {
		MYASSERT(false);
	}
}

bool Recompiler::clearDRB(size_t index) {
	if(g::verbose) {
		RDEGUB("Clearing DRB %i\n", index);
	}
	DynaRecBlock &drb = drbv[index];

	/*//restore the code... should we do it? should we be selective?
	memcpy(recmem.getp_translated(drb.start_ia, drb.rcbtb_size()*4), drb.ppc_code(),
	drb.rcbtb_size()*4);*/
	//let's try to be selective
	//DWORD *p = (DWORD*)recmem.getp_translated(drb.start_ia(), drb.rcbtb_size()*4);
	//DWORD *p = (DWORD*)(memory + drb.start_pia());
	for(size_t i=0; i<drb.rcbtb_size(); i++) {
		DWORD pia = drb.start_pia() + i*4;
		DWORD temp_opcode = recmem.read_pia(pia);
		if(IS_RECOMPILED_OPCODE_ANYTYPE(temp_opcode)) {
			remove_drb_from_overwrite(pia, index);
		}
	}

	//if(drb.p != NULL)
	//drb.memory_pool.release(drb.p);
	//drb._rcb = SharedContainer<BYTE>(0);
	drb._start_eia = drb._start_pia = drb._rcbtb_size = 0;

	for(list<RefBlock>::iterator i=drb.their_refs.begin(); i!=drb.their_refs.end(); i++)
	{
		MYASSERT(i->drbv_index != index);
		drbv[i->drbv_index].clearReferencesToBlock(index);
		for(size_t j=0; j<i->refs.size(); j++) {
			makeStaticBranchUnknown(i->refs[j]);
		}
	}
	drb.their_refs.clear();

	for(list<size_t>::iterator i=drb.my_refs.begin(); i!=drb.my_refs.end(); i++) {
		MYASSERT(*i != index);
		drbv[*i].clearReferencesFromBlock(index);
	}
	drb.my_refs.clear();

	return true;
}
void Recompiler::makeStaticBranchUnknown(DWORD ref_pos) {
	MYASSERT(!g::rec_onestep);
	*(DWORD*)ref_pos = m_staticblock.getFuncPos(SF_STATICBRANCHHANDLER);
	//REALLY 3vil hack. it should change the JMP into a CALL
	*(BYTE*)(ref_pos + 5) = 0xD0;
}
void Recompiler::makeStaticBranchKnown(DWORD ref_pos, DWORD target_pos) {
	MYASSERT(!g::rec_onestep);
	*(DWORD*)ref_pos = target_pos;
	*(BYTE*)(ref_pos + 5) = 0xE0;
}
void Recompiler::DynaRecBlock::clearReferencesFromBlock(size_t index) {
	for(list<RefBlock>::iterator i=their_refs.begin(); i!=their_refs.end(); i++) {
		if(i->drbv_index == index) {
			their_refs.erase(i);
			break;
		}
	}
}
void Recompiler::DynaRecBlock::clearReferencesToBlock(size_t index) {
	for(list<size_t>::iterator i=my_refs.begin(); i!=my_refs.end(); i++) {
		if(*i == index) {
			my_refs.erase(i);
			break;
		}
	}
}
void Recompiler::addReference(size_t from, size_t to, DWORD refpos) {
	MYASSERT(from < drbv.size());
	MYASSERT(to < drbv.size());
	DynaRecBlock &drb_from = drbv[from];
	DynaRecBlock &drb_to = drbv[to];

	//add "to" reference to "from" block if nonexistant
	bool temp = true;
	for(list<size_t>::iterator i=drb_from.my_refs.begin(); i!=drb_from.my_refs.end(); i++) {
		if(*i == to) {
			temp = false;
			break;
		}
	}
	if(temp) {
		drb_from.my_refs.push_back(to);
	}

	//add "from" reference to "to" block if nonexistant
	temp = true;
	list<RefBlock>::iterator rb;
	for(rb=drb_to.their_refs.begin(); rb!=drb_to.their_refs.end(); rb++) {
		if(rb->drbv_index == from) {
			temp = false;
			break;
		}
	}
	if(temp) {
		RefBlock block;
		block.drbv_index = from;
		drb_to.their_refs.push_back(block);
		rb = drb_to.their_refs.end();
		rb--;
	}

	//add pos reference
	rb->refs.push_back(refpos);
}

bool Recompiler::recompile(DWORD start_eia, DWORD end_eia) {
	static int periods = 0;
	TGLE(PostMessage(hWnd, LM_RECOMPILATION_START, 0, 0));
	LARGE_INTEGER start_time, stop_time;
	//DWORD start_pia = TRANSLATE_IA(start_eia);
	RecBuf recbuf(start_eia, TRANSLATE_IA(start_eia));
	//this should ensure cur_tdrb's null-ness when recompile() ends for whatever reasion
	KEEP_POINTER(cur_recbuf, recbuf);
	//KEEP_POINTER(cur_recbuf, recbuf);
	MYASSERT(cur_recbuf->recbuf_size() == 0);

	RDEGUB("\nRecompiling period %i, starting at 0x%08X, should stop before 0x%08X\n",
		++periods, start_eia, end_eia);

	size_t drbv_index = drbv.size();
	//DRBs may not overlap. This new DRB must not overlap an existing DRB unless
	//their starting points are the same, in which case the old one will be overwritten.
	for(size_t i=0; i<drbv.size(); i++) {
		const DynaRecBlock &drb = drbv[i];
		if(start_eia == drb.start_eia()) {
			TGLE(clearDRB(i));
			drbv_index = i;
			break;
		} else {
			//if drb crosses start_ia
			//MYASSERT(!(drb.start_ia() > start_ia && drb.start_ia() <= end_ia ||
			//drb.end_ia() > start_ia && drb.end_ia() <= end_ia));
			//MYASSERT(!((drb.start_pia() < start_pia && drb.end_pia() > start_pia) ||
			//(drb.start_pia() > start_pia && drb.start_pia() < TRANSLATE_IA(end_eia))));
			if((drb.start_eia() < start_eia && drb.end_eia() > start_eia) ||
				(drb.start_eia() > start_eia && drb.start_eia() < end_eia))
			{
				throw rec_fatal_exception("Internal DynaRec error no.11");
			}
		}
	}

	//Main loop
	rec_cycles=0;
	nia = start_eia;
	bool stop = false;
	last_instruction_type = IT_NORMAL;
	TGLE(QueryPerformanceCounter(&start_time));
	while(!stop && nia < end_eia) {
		if(quit()) {
			//FAIL(rec_abort_exception);
			throw rec_abort_exception("");
		}

		void (Recompiler::*opfunc)() = NULL;
		const char *opasm;

		cia = nia;
		nia += 4;

		bool ipage_fault = false;
		try {
			//this one may throw a page fault exception
			DWORD pia = TRANSLATE_IA(cia);
			{
				static DWORD old_pia;
				if(cia != start_eia) if(pia != old_pia + 4)
					throw rec_fatal_exception("Internal DynaRec error no.9");
				old_pia = pia;
			}
			//this one may throw a page fault exception
			opcode = getPPCCode(pia);
			opfunc = rcarr.getOpFunc(opcode);
		} catch(page_fault_exception) {
			ipage_fault = true;
		}

		TGLE(addcia2pos());
		cur_recbuf->ppc_code.push_back(opcode);

		try {
			if(ipage_fault) {
				//mov eax, eia2pos(cia)
				AB(0xB8);
				AD(eia2pos(cia));
				add_fatal_error(REC_IPAGE_FAULT);
				stop = true;
			} else if(!opfunc) {
				if(g::rec_disasm) {
					DEGUB("%08X  Undefined opcode: 0x%08X "
						"Primary:%i Sec21:%i Sec22:%i Sec25:%i Sec26:%i\n",
						cia, opcode, (opcode & 0xFC000000) >> 26,
						getbitsw(opcode, 21, 30), getbitsw(opcode, 22, 30),
						getbitsw(opcode, 25, 30), getbitsw(opcode, 26, 30));
				}
				//mov eax, eia2pos(cia)
				AB(0xB8);
				AD(eia2pos(cia));
				add_fatal_error(REC_UNDEFINED_OPCODE);
				stop = true;
			} else {
				if(g::rec_disasm) {
					opasm = rcarr.getOpAsm(opcode);
					disassembly::opcode = opcode;
					disassembly::cia = cia;
					Container<char> buffer(1000);
					(*asmarr.getOpFunc(opcode))(buffer);
					DEGUB("%08X  %08X %s%s\n", cia, opcode, opasm, buffer);
				}
				last_instruction_type = IT_NORMAL;
				lrv.clear();
				lv.clear();

				//inc edi
				AB(0x47);
				MYASSERT(opfunc != NULL);
				(this->*opfunc)();

				if(last_instruction_type != IT_BRANCH_UNCONDITIONAL &&
					last_instruction_type != IT_ERROR)
				{
					if(!g::rec_onestep) {
						//To get out of long loops after an arbitrary number of cycles:
						//Exit point, doesn't work with branching instructions or fatal errors.
						//Special handling is used there.

						if(nia == end_eia) {
							//eax = nia_pos
							//mov eax, eia2pos(nia)
							AB(0xB8);
							AD(eia2pos(nia));
							//return	//we could do a JMP here, but this way is safer, I think.
							AB(0xC3);
						} else if(next_instruction_is_static_branch()) {
							//We want to make sure that instructions which jump inside the recompiled code
							//doesn't get the opportunity to trap the execution.
							//AFAICF this is only static branches (b & bc).

							//bt edi, BLOCK_SIZE_EXPONENT  //14^2 => 16 K
							AB(0x0F);
							AB(0xBA);
							AB(0xE7);
							AB(BLOCK_SIZE_EXPONENT);
							//jnc continue
							AB(0x73);
							request_internal_label("continue");
							{ //get_out:
								//eax = nia_pos
								//mov eax, eia2pos(nia)
								AB(0xB8);
								AD(eia2pos(nia));
								//return
								AB(0xC3);
							}
							add_internal_label("continue");
							clear_internal_labels();
						}
					} else {  //g::rec_onestep
						//eax = nia_pos
						//mov eax, eia2pos(nia)
						AB(0xB8);
						AD(eia2pos(nia));
						////fwait
						//AB(0x9B);
						//return
						AB(0xC3);
					}
				}
			}
		} catch(bad_form_exception &e) {
			//This will overwrite any and all previous recompiled code for this instruction.
			//It should allow for graceful handling of exceptions anywhere in the recompiler
			//opcodes and their helpers.
			cur_recbuf->step_back(cur_recbuf->rcbtb[(cia - start_eia) / 4]);

			if(g::rec_disasm) {
				DEGUB("  %08X(%08X)  bad_form_exception: %s\n",
					cia, POS-1, e.what());
			}
			//last_undefined = true;
			stop = true;

			//mov eax, cur_ttb[(cia - cur_sia) / 4	//eia2pos only works on completed rec-blocks
			//eia2pos should work here too...
			//mov eax, eia2pos(cia)
			AB(0xB8);
			AD(eia2pos(cia));
			if(e.type == e.Invalid)
				add_fatal_error(REC_INVALID_INSTRUCTION_FORM);
			else	
				add_fatal_error(REC_UNEMULATED_INSTRUCTION_FORM);
		} catch(rec_fatal_exception &e) {
			DEGUB("rec_fatal_exception: %s\n", e.what());
			FAIL(UE_REC_INTERNAL_ERROR);
		}

		rec_cycles++;

		//MYASSERT(lrv.empty());
		if(!lrv.empty()) {
			for(size_t i=0; i<lrv.size(); i++) {
				DEGUB("Unhandled label request: \"%s\" @ pos %i\n", lrv[i].label, lrv[i].pos);
			}
			FAIL(UE_REC_INTERNAL_ERROR);
		}
	}
	TGLE(QueryPerformanceCounter(&stop_time));

	total_rec_cycles += rec_cycles;

	RDEGUB("Stopped just before 0x%08X\n", start_eia + recbuf.rcbtb.size() * 4);
	DWORD elapsed_time = DWORD(stop_time.QuadPart - start_time.QuadPart);
	RDEGUB("Rec period %i complete. Ran %i cycles in %i ms (%s).\n",
		periods, rec_cycles, elapsed_time, ips_string(rec_cycles, elapsed_time).c_str());
	if(g::verbose && g::rec_log) {
		DEGUB("asm size: %i bytes\n", recbuf.recbuf_size());
		DEGUB("Requests: %i call, %i eia2pos\n",
			recbuf.rcv().size(), recbuf.trv.size());
		DEGUB("RCB Translation Buffer size: %i DWORDs\n", recbuf.rcbtb.size());
	}
	{
		bool rcbtb_ok = true;
		if(recbuf.rcbtb.size() == 0) {
			DEGUB("Big Phat RCB Zero!\n");
			rcbtb_ok = false;
		} else for(DWORD i=1, rcbt=recbuf.rcbtb[0]; i<recbuf.rcbtb.size(); i++) {
			if(rcbt > recbuf.rcbtb[i]) {
				DEGUB("rcbtb error: %i(0x%08X) > %i(0x%08X)\n",
					i-1, rcbt, i, recbuf.rcbtb[i]);
				rcbtb_ok = false;
			} else {
				rcbt = recbuf.rcbtb[i];
			}
		}
		if(rcbtb_ok) {
			VRDEGUB("rcb translation buffer is ok\n");
		} else {
			FAIL(UE_REC_INTERNAL_ERROR);
		}
	}

	//begin copy from below
	if(g::rec_log && (g::dump_rec_bin_asm || g::verbose)) {
		if(g::dump_rec_bin_asm) {
			ostringstream str;
			str << "recb" << setw(3) << setfill('0') << drbv_index;
			dumpRecBlock(recbuf, str.str());
			if(g::rec_disasm) {
				for(size_t i=0; i<recbuf.rcbtb.size(); i++) {
					DEGUB("%08X => %08X\n", recbuf.start_eia + i*4, recbuf.rcbtb[i]);
				}
			}
		}
	}
	//end copy

	//Finish up by adding a drb to the drbv
	MYASSERT(cur_recbuf->rcbtb.size() == cur_recbuf->ppc_code.size());

	const DynaRecBlock drb(recbuf);
	recbuf.step_back(0);

	int filled_trs=0, unknown_branches=0;
	for(size_t i=0; i<cur_recbuf->trv.size(); i++) {
		const IAPOS &tr = cur_recbuf->trv[i];
		DWORD offset = (tr.ia - cur_recbuf->start_eia) / 4;
		if(offset < drb.rcbtb_size()) {
			MAKE(DWORD, drb.rcb()[tr.pos]) = drb.rcbtb()[offset];
			filled_trs++;
		} else {
			//since no other blocks have been created recently... ;)
			//and only static branches use eia2pos with potentially out-of-block addresses...
			makeStaticBranchUnknown(DWORD(drb.rcb() + tr.pos));
			//of course, this could very well blow up in our faces :)
			unknown_branches++;
		}
	}
	VRDEGUB("%i filled TRs | %i unknown branches\n", filled_trs, unknown_branches);

	if(g::rec_log && (g::dump_rec_bin_asm || g::verbose)) {
		if(g::dump_rec_bin_asm) {
			ostringstream str;
			str << "rec" << setw(3) << setfill('0') << drbv_index;
			string basename = str.str();
			dumpRecBlock(drb, str.str());
			if(g::rec_disasm) {
				for(size_t i=0; i<drb.rcbtb_size(); i++) {
					DEGUB("%08X => %08X\n", drb.start_eia() + i*4, drb.rcbtb()[i]);
				}
			}
		}
	}

	if(drbv_index == drbv.size()) {
		drbv.push_back(drb);
		VRDEGUB("DRB %i added\n", drbv_index);
	} else {
		drbv[drbv_index] = drb;
		VRDEGUB("DRB %i replaced\n", drbv_index);
	}
	if(g::verbose && g::rec_disasm) {
		//dumpDRBV
		for(size_t i=0; i<drbv.size(); i++) {
			const DynaRecBlock &idrb = drbv[i];
			DEGUB("DRB %i: 0x%08X(0x%08X)-0x%08X, %i instructions, %i bytes\n", i,
				idrb.start_eia(), idrb.start_pia(), idrb.end_eia(), idrb.rcbtb_size(),
				idrb.recbuf_size());
		}
	}

	RDEGUB("Recompilation completed after %I64i run cycles\n", getCycles());
	TGLE(PostMessage(hWnd, LM_RECOMPILATION_FINISH, 0, 0));
	return true;
}

Recompiler::DynaRecBlock::DynaRecBlock(const RecBuf &recbuf) :
DynaRecBlockBase(recbuf, recbuf.rcbtb.size() * sizeof(DWORD) * 2),
_start_eia(recbuf.start_eia), _start_pia(recbuf.start_pia),
_rcbtb_size(recbuf.rcbtb.size()) {
	for(size_t i=0; i<recbuf.rcbtb.size(); i++) {
		rcbtb()[i] = recbuf.rcbtb[i] + DWORD(rcb());
		ppc_code()[i] = recbuf.ppc_code[i];
	}
}

//DWORD: function_pointer - DWORD(recbuf.p()) - (pos + 4)
//void Recompiler::request_call(DWORD function_pointer) {
//cur_recbuf->request_call(function_pointer);
//}

bool Recompiler::running_rec(DWORD eia) {
	//Timing timing("Running recompilation");
	size_t index = findDRBIndexByEIA(eia);
	DWORD start_eia = (index == -1) ? eia : drbv[index].start_eia();
	TGLE(recompile(start_eia, get_end_eia(start_eia)));
	if(drbv_eia2pos(eia) == 0) {
		TGLE(recompile(eia, get_end_eia(eia)));
	}
	return true;
}

//tricksy one, I know, but it should work.
DWORD Recompiler::get_end_eia(DWORD start_eia) {
	DWORD end_eia = BAD_ADDRESS;
	for(size_t i=0; i<drbv.size(); i++) {
		if(drbv[i].start_eia() < end_eia && drbv[i].start_eia() > start_eia)
			end_eia = drbv[i].start_eia();
	}
	return end_eia;
}

size_t Recompiler::get_rcbtb_index(const DynaRecBlock &drb, DWORD h4xx0r_pos) {
	MYASSERT(h4xx0r_pos >= DWORD(drb.rcb()) &&
		h4xx0r_pos < DWORD(drb.rcb()) + drb.recbuf_size() &&
		DWORD(drb.rcb()) == drb.rcbtb()[0]);
	size_t i;
	for(i=1; i<drb.rcbtb_size(); i++) {
		if(drb.rcbtb()[i] > h4xx0r_pos)
			break;

	}
	return i-1;
}

bool Recompiler::do_nia_pos(DWORD current_ia, DWORD requested_nia, bool allow_overwrite) {
	//find block containing requested_nia
	nia_pos = raw_eia2pos(requested_nia);
	if(nia_pos == 0) {
		size_t index = findDRBIndexByEIA(requested_nia);
		if(index != -1) {
			const DynaRecBlock &drb = drbv[index];
			size_t rcbtv_index = (requested_nia - drb.start_eia()) / 4;
			DWORD pia = TRANSLATE_IA(requested_nia);
			DWORD temp_opcode = recmem.read_pia(pia);
			if(temp_opcode == drb.ppc_code()[rcbtv_index]) {
				if(allow_overwrite)
					add_drb2overwrite(pia, index);
				nia_pos = drb.rcbtb()[rcbtv_index];
			}	//else recompile
		}
	}
	//if not found, recompile
	if(nia_pos == 0) {
		if(g::rec_log && g::verbose) {
			DEGUB("Branch or interrupt @ ");
			if(current_ia == BAD_ADDRESS) {
				DEGUB("unknown address");
			} else {
				DEGUB("0x%08X", current_ia);
			}
			DEGUB(" caused recompilation @ 0x%08X\n", requested_nia);
		}
		TGLE(running_rec(requested_nia));
		nia_pos = drbv_eia2pos(requested_nia);
		MYASSERT(nia_pos != 0);
	}
	return true;
}

/*bool memeq_swapw(const void* a, const void* b, size_t len) {
MYASSERT((len & 3) == 0);
DWORD* da = (DWORD*)a;
DWORD* db = (DWORD*)b;
size_t s = len >> 2;
for(size_t i=0; i<s; i++) {
if(da[i] != swapw(db[i]))
return false;
}
return true;
}*/
DWORD memcmp_swapw(const void* a, const void* b, size_t len) {
	MYASSERT((len & 3) == 0);
	DWORD* da = (DWORD*)a;
	DWORD* db = (DWORD*)b;
	size_t s = len >> 2;
	for(size_t i=0; i<s; i++) {
		if(da[i] != swapw(db[i]))
			return i;
	}
	return (DWORD)-1;
}
struct REQUEST { DWORD index, lo, hi; };
bool Recompiler::doICBI() {
	DWORD current_eia = nia_pos;
	DWORD current_drbi = findDRBIndexByEIA(current_eia);
	DynaRecBlock& current_drb = drbv[current_drbi];
	nia_pos = current_drb.eia2pos(current_eia + 4); //important
	opcode = current_drb.ppc_code()[(current_eia - current_drb.start_eia()) / 4];
	D_rA; D_rB;
	DWORD requested_eia = rA_0 + r.gpr[rB];
	VRDEGUB("ICBI @ 0x%08X: 0x%08X\n", current_eia, requested_eia);
	requested_eia &= CACHE_BLOCK_MASK;
	//DWORD requested_drbi = findDRBIndexByEIA(requested_eia);
	//ideally, find all blocks which intersect the requested cache block.
	vector<REQUEST> request_v;
	for(size_t i=0; i<drbv.size(); i++) {
		DynaRecBlock& drb = drbv[i];
		//nice intersect algo
		DWORD lo = MAX(drb.start_eia(), requested_eia);
		DWORD hi = MIN(drb.end_eia(), requested_eia + CACHE_BLOCK_SIZE);
		if(lo < hi) {
			REQUEST req = { i, lo, hi };
			request_v.push_back(req);
		}
	}
	if(request_v.size()) {
		VRDEGUB("%i blocks:\n", request_v.size());
	}
	for(size_t i=0; i<request_v.size(); i++) {
		REQUEST& request = request_v[i];
		VRDEGUB("%i ", request.index);
		if(current_drbi == request.index)
			throw rec_fatal_exception("Unimplemented ICBI type 1!\n");
		DynaRecBlock& drb = drbv[request.index];

		DWORD offset = request.lo - drb.start_eia();
		DWORD size = request.hi - request.lo;
		const void* mem = recmem.getp_physical(drb.start_pia() + offset, size);
		const void* cache = drb.ppc_code() + (offset >> 2);
		DWORD res;
		if((res = memcmp_swapw(mem, cache, size)) != -1) {
			VRDEGUB("pia 0x%08X\n", drb.start_pia() + offset + (res << 2));
			//clearDRB(request.index);	//may be flawed in this context
			TGLE(running_rec(drb.start_eia()));
		}
	}
	if(request_v.size()) {
		VRDEGUB("\n");
	}
	return true;
}

//Tentative standard interrupt emulation
bool Recompiler::take_interrupt(DWORD requested_nia) {
	TGLE(do_nia_pos(BAD_ADDRESS, requested_nia, true));
	return true;
}

void Recompiler::function_container(RecFunc function) {
	_flushall();
	DWORD temp_cycles, temp_nia_pos;
	//Timing timing("RecCode");
	__asm {
#ifndef _DEBUG
		pusha //see notes.txt
#endif
			xor edi, edi  //We use edi as a cycle counter.
			call function
			mov temp_cycles, edi
			mov temp_nia_pos, eax
#ifndef _DEBUG
			popa
#endif
	}
	cc.addCycles(temp_cycles);
	nia_pos = temp_nia_pos;
	//DEGUB("%i cycles of ", temp_cycles);
}

bool Recompiler::_run(DWORD runstop, bool degub, QWORD degub_skip, bool dumpmem) {
	DWORD start_time, stop_time;
	QWORD rec_function_calls=0;
	DWORD min_cycles_per_call=MAX_DWORD, max_cycles_per_call=0;
	bool error = false;

	bool temp_degub=false;

	start_time = GetTickCount();  //Record start time
	try { do {
		errorcode = REC_NO_ERROR;
		//recmem.mdegub_on = false;

		while(!quit() && !error) {
			if(g::rec_onestep) {
				temp_degub = (degub && getCycles() >= degub_skip) ||
					(g::lowmem_log && PHYSICALIZE(reverse_rcbt(nia_pos)) < 0x3100);
				if(temp_degub) {
					//recmem.mdegub_on = true;
					DEGUB("  %08X(%08X)\n", reverse_rcbt(nia_pos), nia_pos);
				}
			}
			//DEGUB("  %08X(%08X)\n", reverse_rcbt(nia_pos), nia_pos);	//temp

			MYASSERT(errorcode == REC_NO_ERROR);
			{
				QWORD cycles_before_call = cc.cycles;
				rec_function_calls++;
				function_container((RecFunc)(nia_pos));	//the center of the universe ;)
				DWORD cycles_this_call = DWORD(cc.cycles - cycles_before_call);
				MYASSERT((int)cycles_this_call >= 0);
				min_cycles_per_call = MIN(min_cycles_per_call, cycles_this_call);
				max_cycles_per_call = MAX(max_cycles_per_call, cycles_this_call);
			}
			if(nia_pos == m_staticblock.getFuncPos(SF_STATICBRANCHHANDLER)) {
				DEGUB("Here we are.\n");
			}

			if(errorcode != REC_NO_ERROR)
				error = true;
			if(!error) {
				TGLE(do_interrupts());
			}

			if(g::rec_onestep) {
				temp_degub = (degub && getCycles() >= degub_skip) ||
					(g::lowmem_log && PHYSICALIZE(reverse_rcbt(nia_pos)) < 0x3100);
				if(temp_degub) {
					recmem.degub();
					r.degub_tb_dec(old_registers);
					r.degub(false, old_registers);
				}
			}
			if(RUNSTOP && cc.cycles_current_period >= runstop)
				break;
		}

		if(error) {
			if(errorcode == REC_BCLR || errorcode == REC_BCCTR || errorcode == REC_RFI) {
				//this should be as fast as possible

				DWORD current_eia = nia_pos;
				DWORD requested_nia;
				switch(errorcode) {
				case REC_BCLR:
					requested_nia = r.lr;
					break;
				case REC_BCCTR:
					requested_nia = r.ctr;
					break;
				case REC_RFI:
					requested_nia = r.srr0;
					break;
				default:
					FAIL(UE_REC_INTERNAL_ERROR);
				}
				//save it now because of possible recompilation which may alter anything
				DWORD temp_opcode = 0;
				bool not_rfi = errorcode != REC_RFI;
				if(not_rfi) {
					size_t index = getDRBIndexRaw(current_eia);
					if(index == BAD_ADDRESS) {
						index = findDRBIndexByEIA(current_eia);
					}
					MYASSERT(index < drbv.size());
					const DynaRecBlock &drb = drbv[index];
					temp_opcode = drb.ppc_code()[(current_eia - drb.start_eia()) / 4];
				}

				TGLE(do_nia_pos(current_eia, requested_nia, true));

				if(not_rfi) {
					//Take over the rest of the BC*R's jump job
					opcode = temp_opcode;
					D_LK;
					if(LK) {
						r.lr = current_eia + 4;
					}
				}
				error = false;
			} else if(errorcode == REC_ICBI) {
				TGLE(doICBI());
				error = false;
			} else if(errorcode == REC_STATIC_BRANCH) {
				//we can take our time here
				DWORD h4xx0r_pos = nia_pos - 6;	//dark magik

				size_t from_index = findDRBIndexByPos(h4xx0r_pos); //will work
				MYASSERT(from_index != -1);
				DynaRecBlock &drb_from = drbv[from_index];
				size_t rcbtbi = get_rcbtb_index(drb_from, h4xx0r_pos);
				DWORD current_eia = drb_from.start_eia() + rcbtbi*4;
				opcode = drb_from.ppc_code()[rcbtbi];

				DWORD requested_nia;
				void (Recompiler::*opfunc)() = rcarr.getOpFunc(opcode);
				if(opfunc == &Recompiler::_b) {
					D_LI; D_AA;
					requested_nia = AA ? LI : current_eia + LI;
				} else if(opfunc == &Recompiler::_bc) {
					D_BD; D_AA;
					requested_nia = AA ? BD : current_eia + BD;
				} else {
					FAIL(UE_REC_INTERNAL_ERROR);
				}
				TGLE(do_nia_pos(current_eia, requested_nia, false));

				MYASSERT(from_index == findDRBIndexByPos(h4xx0r_pos));
				size_t to_index = findDRBIndexByEIA(requested_nia);
				DynaRecBlock &drb_to = drbv[to_index];
				DWORD target_pos = drb_to.rcbtb()[(requested_nia - drb_to.start_eia()) / 4];
				makeStaticBranchKnown(h4xx0r_pos, target_pos);
				addReference(from_index, to_index, h4xx0r_pos);
				error = false;
			} else if(errorcode == REC_ONESTEP_STATIC_BRANCH) {
				TGLE(do_nia_pos(BAD_ADDRESS, nia_pos, false));
				error = false;
			} else if(errorcode == REC_IPAGE_FAULT) {
				recmem.raiseISI();
				error = false;
			} else if(g::advanced_mode && errorcode == REC_INVALID_MEMORY_ADDRESS) {
				DWORD h4xx0r_pos = nia_pos;
				size_t index = findDRBIndexByPos(h4xx0r_pos);
				MYASSERT(index != -1);
				DynaRecBlock &drb = drbv[index];
				nia_pos = drb.rcbtb()[get_rcbtb_index(drb, h4xx0r_pos)];

				interrupt.raise(INTERRUPT_MACHINECHECK);
				DEGUB("Machine check exception raised @ 0x%08X\n",
					drb.start_eia() + get_rcbtb_index(drb, h4xx0r_pos)*4);
				error = false;
			} else
				break;
		}
		if(!error) {
			TGLE(do_interrupts());
		}
	} while(!(quit() || (RUNSTOP && cc.cycles_current_period >= runstop)));
	}
#define CATCH(classname) catch(classname &e) {\
	recmem.degub();\
	if(temp_degub) {\
	r.degub_tb_dec(old_registers);\
	r.degub(false, old_registers);\
	}\
	g_errorstring = e.what();\
	DEGUB("%s: %s\n", #classname, g_errorstring.c_str());\
	error = true;\
	errorcode = REC_EXCEPTION; }
	CATCH(bouehr_exception) CATCH(generic_fatal_exception) CATCH(hardware_fatal_exception)
		CATCH(rec_fatal_exception) CATCH(emulation_stop_exception)
		catch(rec_abort_exception)
	{
		recmem.degub();
		if(temp_degub) {
			r.degub_tb_dec(old_registers);
			r.degub(false, old_registers);
		}
		DEGUB("rec_abort_exception\n");
		error = false;
	}

	stop_time = GetTickCount();	//Record stop time

	TGLE(h.killPeriodicTasks(error ? NULL : &g_errorstring));
	if(!g_errorstring.empty()) {
		error = true;
		errorcode = REC_NO_ERROR;
	}
	DWORD elapsed_time = stop_time - start_time;
	//if(!g::dual_run) {
	DEGUB("Run period %i complete. Ran %I64i cycles in %i ms (%s).\n",
		periods, cc.cycles_current_period, elapsed_time,
		ips_string(cc.cycles_current_period, elapsed_time).c_str());
	DEGUB("%I64i rec function calls (%.f cycles/call)\n",
		rec_function_calls, cc.cycles_current_period/(double)rec_function_calls);
	DEGUB("cycles/call: min %i | max %i\n", min_cycles_per_call, max_cycles_per_call);
	//}
	cc.endPeriod();

	if(dumpmem)
		recmem.dumpmem();

	ostringstream str;
	if(error) switch(errorcode) {
	case REC_NO_ERROR:
		break;
	case REC_INFINITE_LOOP:
		str << "Infinite loop detected at " << HEX08(reverse_rcbt(nia_pos)) <<
			"(" << HEX08(nia_pos) << ")";
		break;
	case REC_UNDEFINED_OPCODE:
		str << "Undefined opcode at " << HEX08(reverse_rcbt(nia_pos)) <<
			"(" << HEX08(nia_pos) << ")";
		break;
	case REC_UNEMULATED_INSTRUCTION:
		str << "Unemulated instruction at " << HEX08(reverse_rcbt(nia_pos)) <<
			"(" << HEX08(nia_pos) << ")";
		break;
	case REC_UNIMPLEMENTED_OPCODE:
		str << "Unimplemented opcode at " << HEX08(reverse_rcbt(nia_pos)) <<
			"(" << HEX08(nia_pos) << "): " << rcarr.getOpAsm(recmem.rw(reverse_rcbt(nia_pos)));
		break;
	case REC_INVALID_INSTRUCTION_FORM:
		str << "Invalid instruction form at " << HEX08(reverse_rcbt(nia_pos)) <<
			"(" << HEX08(nia_pos) << ")";
		break;
	case REC_UNEMULATED_INSTRUCTION_FORM:
		str << "Unemulated instruction form at " << HEX08(reverse_rcbt(nia_pos)) <<
			"(" << HEX08(nia_pos) << ")";
		break;
	case REC_IPAGE_FAULT:
		str << "Internal error: Page faulted instruction at " <<
			HEX08(reverse_rcbt(nia_pos)) << "(" << HEX08(nia_pos) << ")";
		break;
	case REC_INVALID_MEMORY_ADDRESS:
		str << "Invalid memory address: " <<
			HEX08(nia_pos - DWORD(memory)) << "(" << HEX08(nia_pos) << ")";
		break;
	case REC_INTERNAL_ERROR:
		str << "Internal error at or near " << HEX08(reverse_rcbt(nia_pos)) <<
			"(" << HEX08(nia_pos) << "): " << g_errorstring;
		break;
	case REC_EXCEPTION:
		break;
	default:
		DEGUB("nia_pos=%08X\n", nia_pos);
		str << "Unknown internal error " << errorcode << ".";
	}
	if(!str.str().empty())
		g_errorstring = str.str();
	SetLastError(NO_ERROR); //This signifies that g_errorstring has been loaded
	return !error;
}
